/*
	Matrix factorisation routines to work with the other matrix files.
*/

/* LUfactor.c 1.5 11/25/87 */
static	char	rcsid[] = "$Header: /usr/local/home/des/meschach/meschach/RCS/lufactor.c,v 1.5 1992/02/16 22:00:09 des Exp $";

#include	<stdio.h>
#include	<math.h>
#include	"matrix.h"

/* Most matrix factorisation routines are in-situ unless otherwise specified */

static	double	*scale = (double *)NULL;
static	int	sz_scale = 0;

/* LUfactor -- gaussian elimination with scaled partial pivoting
		-- Note: returns LU matrix which is A */
MAT	*LUfactor(A,pivot)
MAT	*A;
PERM	*pivot;
{
	u_int	i, j, k, k_max, m, n;
	int	i_max;
	double	max1, **A_v, *A_piv, *A_row, temp;

	if ( A==(MAT *)NULL || pivot==(PERM *)NULL )
		error(E_NULL,"LUfactor");
	if ( pivot->size != A->m )
		error(E_SIZES,"LUfactor");
	m = A->m;	n = A->n;
	if ( scale == (double *)NULL || sz_scale < m )
	{	/* get scale array */
		if ( scale != (double *)NULL )
			free(scale);
		if ((scale=NEW_A(m,double)) == (double *)NULL)
			error(E_MEM,"LUfactor");
		sz_scale = m;
	}
	A_v = A->me;

	/* initialise pivot with identity permutation */
	for ( i=0; i<m; i++ )
		pivot->pe[i] = i;

	/* set scale parameters */
	for ( i=0; i<m; i++ )
	{
		max1 = 0.0;
		for ( j=0; j<n; j++ )
		{
			temp = fabs(A_v[i][j]);
			max1 = max(max1,temp);
		}
		scale[i] = max1;
	}

	/* main loop */
	k_max = min(m,n)-1;
	for ( k=0; k<k_max; k++ )
	{
		/* find best pivot row */
		max1 = 0.0;	i_max = -1;
		for ( i=k; i<m; i++ )
			if ( scale[i] > 0.0 )
			{
				temp = fabs(A_v[i][k])/scale[i];
				if ( temp > max1 )
				{ max1 = temp;	i_max = i;	}
			}

		/* if no pivot then ignore column k... */
		if ( i_max == -1 )
			continue;

		/* do we pivot ? */
		if ( i_max != k )	/* yes we do... */
		{
			trans_px(pivot,i_max,k);
			for ( j=0; j<n; j++ )
			{
				temp = A_v[i_max][j];
				A_v[i_max][j] = A_v[k][j];
				A_v[k][j] = temp;
			}
		}

		/* row operations */
		for ( i=k+1; i<m; i++ )	/* for each row do... */
		{	/* Note: divide by zero should never happen */
			temp = A_v[i][k] = A_v[i][k]/A_v[k][k];
			A_piv = &(A_v[k][k+1]);
			A_row = &(A_v[i][k+1]);
			if ( k+1 < n )
			    __mltadd__(A_row,A_piv,-temp,(int)(n-(k+1)));
			/*********************************************
			for ( j=k+1; j<n; j++ )
				A_v[i][j] -= temp*A_v[k][j];
				(*A_row++) -= temp*(*A_piv++);
			*********************************************/
		}

	}

	return A;
}


/* LUsolve -- given an LU factorisation in A, solve Ax=b */
VEC	*LUsolve(A,pivot,b,x)
MAT	*A;
PERM	*pivot;
VEC	*b,*x;
{
	if ( A==(MAT *)NULL || b==(VEC *)NULL || pivot==(PERM *)NULL )
		error(E_NULL,"LUsolve");
	if ( A->m != A->n || A->n != b->dim )
		error(E_SIZES,"LUsolve");
	if ( b == x )
		error(E_INSITU,"LUsolve");
	if ( x==(VEC *)NULL || x->dim < b->dim )
		x = get_vec(b->dim);
	px_vec(pivot,b,x);	/* x := P.b */
	Lsolve(A,x,x,1.0);	/* implicit diagonal = 1 */
	Usolve(A,x,x,0.0);	/* explicit diagonal */

	return (x);
}

/* LUTsolve -- given an LU factorisation in A, solve A^T.x=b */
VEC	*LUTsolve(LU,pivot,b,x)
MAT	*LU;
PERM	*pivot;
VEC	*b,*x;
{
	static VEC	*tmp=VNULL;

	if ( ! LU || ! b || ! pivot )
		error(E_NULL,"LUTsolve");
	if ( LU->m != LU->n || LU->n != b->dim )
		error(E_SIZES,"LUTsolve");
	if ( b == x )
		error(E_INSITU,"LUTsolve");
	if ( ! x || x->dim < b->dim )
		x = get_vec(b->dim);

	tmp = cp_vec(b,tmp);
	UTsolve(LU,tmp,tmp,0.0);	/* explicit diagonal */
	LTsolve(LU,tmp,tmp,1.0);	/* implicit diagonal = 1 */
	pxinv_vec(pivot,tmp,x);	/* x := P^T.tmp */

	return (x);
}

/* m_inverse -- returns inverse of A, provided A is not too rank deficient
	-- uses LU factorisation */
MAT	*m_inverse(A,out)
MAT	*A, *out;
{
	int	i;
	VEC	*tmp, *tmp2;
	MAT	*A_cp;
	PERM	*pivot;

	if ( ! A )
	    error(E_NULL,"m_inverse");
	if ( A->m != A->n )
	    error(E_SQUARE,"m_inverse");
	if ( ! out || out->m < A->m || out->n < A->n )
	    out = m_resize(out,A->m,A->n);

	A_cp = cp_mat(A,MNULL);
	tmp = get_vec(A->m);
	tmp2 = get_vec(A->m);
	pivot = get_perm(A->m);
	tracecatch(LUfactor(A_cp,pivot),"m_inverse");
	for ( i = 0; i < A->n; i++ )
	{
	    zero_vec(tmp);
	    tmp->ve[i] = 1.0;
	    tracecatch(LUsolve(A_cp,pivot,tmp,tmp2),"m_inverse");
	    set_col(out,i,tmp2);
	}

	freemat(A_cp);
	freevec(tmp);	freevec(tmp2);
	freeperm(pivot);

	return out;
}

/* LUcondest -- returns an estimate of the condition number of LU given the
	LU factorisation in compact form */
double	LUcondest(LU,pivot)
MAT	*LU;
PERM	*pivot;
{
    static	VEC	*y = VNULL, *z = VNULL;
    double	L_norm, U_norm, sum;
    int		i, j, n;

    if ( ! LU || ! pivot )
	error(E_NULL,"LUcondest");
    if ( LU->m != LU->n )
	error(E_SQUARE,"LUcondest");
    if ( LU->n != pivot->size )
	error(E_SIZES,"LUcondest");

    n = LU->n;
    y = v_resize(y,n);
    z = v_resize(z,n);

    for ( i = 0; i < n; i++ )
    {
	sum = 0.0;
	for ( j = 0; j < i; j++ )
	    sum -= LU->me[j][i]*y->ve[j];
	sum -= (sum < 0.0) ? 1.0 : -1.0;
	if ( LU->me[i][i] == 0.0 )
	    return HUGE_VAL;
	y->ve[i] = sum / LU->me[i][i];
    }

    LTsolve(LU,y,y, 0.0); /* bug fix by Jim Dai, 0.0 may be 1.0 */
    LUsolve(LU,pivot,y,z);

    /* now estimate norm of A (even though it is not directly available) */
    /* actually computes ||L||_inf.||U||_inf */
    U_norm = 0.0;
    for ( i = 0; i < n; i++ )
    {
	sum = 0.0;
	for ( j = i; j < n; j++ )
	    sum += fabs(LU->me[i][j]);
	if ( sum > U_norm )
	    U_norm = sum;
    }
    L_norm = 0.0;
    for ( i = 0; i < n; i++ )
    {
	sum = 1.0;
	for ( j = 0; j < i; j++ )
	    sum += fabs(LU->me[i][j]);
	if ( sum > L_norm )
	    L_norm = sum;
    }

    return U_norm*L_norm*v_norm_inf(z)/v_norm_inf(y);
}

